{*******************************************************}
{       This unit is part of eyeOS microserver          }
{                                                       }
{ Author:                                               }
{ Bjrn Ahrens                                          }
{ bjoern@ahrens.net                                     }
{ http://bjoern.ahrens.net                              }
{*******************************************************}

unit SettingsUnit;

interface

type
  TSettings = class
  private
    FHttpPort: Integer;
    FHttpUPnP: Boolean;
    FSmtpPassword: String;
    FSmtpHost: String;
    FSmtpActive: Boolean;
    FSmtpUsername: String;
    FHttpFirewall: Boolean;
    FSmtpNeverDirect: Boolean;
    FProgPath: String;
    FHttpRootPath: String;
    FSmtpServerPort: Integer;
    FPHPUpload: Integer;
    FHTTPIPRanges : Integer;
    FHttpZLib : Boolean;
    FAutoStart : Boolean;
    FLanguage : String;
    FIsOnRemovableDrive : Boolean;
    FCheckForUpdates: Boolean;
    FUseTLS: Boolean;

    FOpenSSLAvailable : Boolean;
    FTLSKeyPwd: String;
    FTLSKeyFile: String;
    FTLSSavePwd: Boolean;
    FTLSCertFile: String;
    FPreferUpdate: Boolean;
  public
    constructor Create();

    procedure Load();
    procedure Save();

    property AutoStart : Boolean read FAutoStart write FAutoStart;
    property Language : String read FLanguage write FLanguage;

    property HttpPort : Integer read FHttpPort write FHttpPort;
    property HttpUPnP : Boolean read FHttpUPnP write FHttpUPnP;
    property HttpFirewall : Boolean read FHttpFirewall write FHttpFirewall;
    property HTTPIPRanges : Integer read FHTTPIPRanges write FHTTPIPRanges;
    property HttpZLib : Boolean read FHttpZlib write FHttpZlib;

    property UseTLS : Boolean read FUseTLS write FUseTLS;
    property OpenSSLAvailable : Boolean read FOpenSSLAvailable;
    property TLS_CertFile : String read FTLSCertFile write FTLSCertFile;
    property TLS_KeyFile : String read FTLSKeyFile write FTLSKeyFile;
    property TLS_KeyPassword : String read FTLSKeyPwd write FTLSKeyPwd;
    property TLS_SavePassword : Boolean read FTLSSavePwd write FTLSSavePwd;

    property SmtpActive : Boolean read FSmtpActive write FSmtpActive;
    property SmtpServerPort : Integer read FSmtpServerPort write FSmtpServerPort;
    property SmtpNeverDirect : Boolean read FSmtpNeverDirect write FSmtpNeverDirect;
    property SmtpHost : String read FSmtpHost write FSmtpHost;
    property SmtpUsername : String read FSmtpUsername write FSmtpUsername;
    property SmtpPassword : String read FSmtpPassword write FSmtpPassword;

    property PHPMaxUpload : Integer read FPHPUpload write FPHPUpload;

    property ProgPath : String read FProgPath;
    property HttpRootPath : String read FHttpRootPath;
    property IsOnRemovableDrive : Boolean read FIsOnRemovableDrive;
    property CheckForUpdates : Boolean read FCheckForUpdates write FCheckForUpdates;

    property eyeOS_PreferUpdate : Boolean read FPreferUpdate write FPreferUpdate;
  end;

  function Settings () : TSettings;

implementation

uses SysUtils, IniFiles, Registry, Windows;

var FSettings : TSettings;

function Settings () : TSettings;
begin
  if FSettings=nil then
    FSettings:=TSettings.Create();
  Result:=FSettings;
end;

{ TSettings }

constructor TSettings.Create;
var         H : THandle;
begin
  FProgpath:=IncludeTrailingPathDelimiter(ExtractFilePath(ParamStr(0)));
  FHttpRootpath:=progpath+'eyeos';

  {berprfen, ob MicroServer von Removable Drive gestartet wird (z.B. von USB-Stick)}
  FIsOnRemovableDrive:=GetDriveType(PChar(ExtractFileDrive(FProgPath))) in [DRIVE_REMOVABLE, DRIVE_CDROM];

  {auf OpenSSL-DLLs prfen}
  FOpenSSLAvailable:=false;

  try
    H:=LoadLibrary(PChar('libeay32.dll'));
    if H<>0 then begin
      FreeLibrary(H);
      H:=LoadLibrary(PChar('ssleay32.dll'));
      if (H<>0) then begin
        FreeLibrary(H);
        FOpenSSLAvailable:=true;
      end;
    end;
  except;
  end;

  Load();
end;

procedure TSettings.Load;
var       Ini : TIniFile;
          Reg : TRegistry;
begin
  Ini:=TIniFile.Create(FProgpath+'settings.ini');
  try
    FLanguage:=Ini.ReadString('misc','language','');
    FCheckForUpdates:=Ini.ReadBool('misc','updates',true);

    FHttpUPnP:=Ini.ReadBool('server','upnp',false);
    FHttpPort:=Ini.ReadInteger('server','port',8080);
    FHttpFirewall:=Ini.ReadBool('server','winfirewall',false);
    FHTTPIPRanges:=Ini.ReadInteger('server','ipranges',0);
    FHttpZLib:=Ini.ReadBool('server','zlib',false);

    FUseTLS:=Ini.ReadBool('server','tls',false) and FOpenSSLAvailable;
    FTLSSavePwd:=Ini.ReadBool('tls','savepwd',true);
    if FTLSSavePwd then
      FTLSKeyPwd:=Ini.ReadString('tls','keypwd','eyeosmicroserver')
    else
      FTLSKeyPwd:='';
    FTLSKeyFile:=Ini.ReadString('tls','keyfile',FProgPath+'default.key');
    FTLSCertFile:=Ini.ReadString('tls','certfile',FProgPath+'default.crt');


    FSmtpHost:=Ini.ReadString('smtp','host','');
    FSmtpUsername:=Ini.ReadString('smtp','username','');
    FSmtpPassword:=Ini.ReadString('smtp','password','');
    FSmtpNeverDirect:=Ini.ReadBool('smtp','neverdirect',true);
    FSmtpActive:=Ini.ReadBool('smtp','active',false);
    FSmtpServerPort:=Ini.ReadInteger('smtp','internalport',25);

    FPHPUpload:=Ini.ReadInteger('php','maxupload',10);

    FPreferUpdate:=Ini.ReadBool('eyeos','preferupdate',true);
  finally
    Ini.Free();
  end;

  if not IsOnRemovableDrive then begin
    Reg:=TRegistry.Create();
    try
      Reg.RootKey:=HKEY_CURRENT_USER;
      if Reg.OpenKey('\Software\Microsoft\Windows\CurrentVersion\Run',false) then begin
        FAutoStart:=Reg.ValueExists('eyeOS MicroServer');
      end;
    finally
      Reg.Free();
    end;
  end else
    FAutoStart:=false;
end;

procedure TSettings.Save;
var       Ini : TIniFile;
          Reg : TRegistry;
begin
  Ini:=TIniFile.Create(FProgPath+'settings.ini');
  try
    Ini.WriteString('misc','language',FLanguage);
    Ini.WriteBool('misc','updates',FCheckForUpdates);
    Ini.WriteBool('server','upnp',FHttpUPnP);
    Ini.WriteInteger('server','port',FHttpPort);
    Ini.WriteString('smtp','host',FSmtpHost);
    Ini.WriteString('smtp','username',FSmtpUsername);
    Ini.WriteString('smtp','password',FSmtpPassword);
    Ini.WriteBool('smtp','neverdirect',FSmtpNeverDirect);
    Ini.WriteBool('smtp','active',FSmtpActive);
    Ini.WriteBool('server','winfirewall',FHttpFirewall);
    Ini.WriteInteger('smtp','internalport',FSmtpServerPort);
    Ini.WriteInteger('php','maxupload',FPHPUpload);
    Ini.WriteInteger('server','ipranges',FHTTPIPRanges);
    ini.WriteBool('server','zlib',FHttpZLib);
    Ini.WriteBool('server','tls',FUseTLS);

    Ini.WriteBool('tls','savepwd',FTLSSavePwd);
    if FTLSSavePwd then
      Ini.WriteString('tls','keypwd',FTLSKeyPwd)
    else
      Ini.DeleteKey('tls','keypwd');
    Ini.WriteString('tls','keyfile',FTLSKeyFile);
    Ini.WriteString('tls','certfile',FTLSCertFile);
    Ini.WriteBool('eyeos','preferupdate',FPreferUpdate);
  finally
    Ini.Free();
  end;

  if not IsOnRemovableDrive then begin
    Reg:=TRegistry.Create();
    try
      Reg.RootKey:=HKEY_CURRENT_USER;
      if Reg.OpenKey('\Software\Microsoft\Windows\CurrentVersion\Run',false) then begin
        if (FAutoStart) then
          Reg.WriteString('eyeOS MicroServer',ParamStr(0))
        else
          Reg.DeleteValue('eyeOS MicroServer');
      end;
    finally
      Reg.Free();
    end;
  end;
end;

initialization
finalization
  FSettings.Free();
end.
