{*******************************************************}
{       This unit is part of eyeOS microserver          }
{                                                       }
{ Author:                                               }
{ Bjrn Ahrens                                          }
{ bjoern@ahrens.net                                     }
{ http://bjoern.ahrens.net                              }
{*******************************************************}

unit WindowsFirewall;
{-------------------------------------------------------------------------------}
{-------------------------------------------------------------------------------}
interface
{-------------------------------------------------------------------------------}
{-------------------------------------------------------------------------------}

uses Contnrs, NetFwTypeLib_TLB;

{-------------------------------------------------------------------------------}

type
  TWindowsFirewall = class
    public
      type TItem = class
        Name       : WideString;
        Enabled    : Boolean;

        constructor Create (AName : WideString);
      end;
      type TApplication = class (TItem)
        ProcessImageFileName : WideString;
        constructor Create (AProcessImageFileName, AName : WideString);
      end;
      type TPort = class (TItem)
        ipProtocol : NET_FW_IP_PROTOCOL_;
        PortNumber : Integer;
        constructor Create (APortNumber : Integer; AnIpProtocol : NET_FW_IP_PROTOCOL_; AName : WideString);
      end;
      type TScope = (wfsAll, wfsLocalSubnet, wfsLocalhost);
    private
      FAutoRollback : Boolean;
      fwProfile     : INetFwProfile;
      Items         : TObjectList;
    protected
      procedure RemoveItem (AItem : TWindowsFirewall.TItem);
      function FindPortItem (portNumber : Integer; ipProtocol : NET_FW_IP_PROTOCOL_) : TPort;
      function FindAppItem (AProcessImageFileName : WideString) : TApplication;
      function GetEnabled : Boolean;
      procedure SetEnabled (Value : Boolean);
    public
      constructor Create();
      destructor Destroy(); override;

      function AppIsEnabled (const ProcessImageFileName : WideString) : Boolean;
      procedure AddApp (const ProcessImageFileName, Name : WideString; AScope : TScope = wfsAll);
      procedure RemoveApp (const ProcessImageFileName : WideString);

      function PortIsEnabled (portNumber : Integer; ipProtocol : NET_FW_IP_PROTOCOL_) : Boolean;
      function AddPort (portNumber : Integer; ipProtocol : NET_FW_IP_PROTOCOL_; Name : WideString; LocalScope : Boolean = false) : TPort;
      procedure RemovePort (portNumber : Integer; ipProtocol : NET_FW_IP_PROTOCOL_); overload;
      procedure RemovePort (APortItem : TPort); overload;

      procedure RollBack ();

      property Enabled : Boolean read GetEnabled write SetEnabled;
      property AutoRollback : Boolean read FAutoRollback write FAutoRollback;
  end;

const
  NET_FW_IP_PROTOCOL_TCP = $00000006;
  NET_FW_IP_PROTOCOL_UDP = $00000011;

{-------------------------------------------------------------------------------}
{-------------------------------------------------------------------------------}
implementation
{-------------------------------------------------------------------------------}
{-------------------------------------------------------------------------------}

uses Windows, ComObj, SysUtils, ActiveX;

{-------------------------------------------------------------------------------}
constructor TWindowsFirewall.Create();
var      fwPolicy : INetFwPolicy;
         fwMgr    : INetFwMgr;
         hr       : HRESULT;
begin
  fwProfile:=nil;
  fwMgr:=nil;
  fwPolicy:=nil;
  FAutoRollBack:=true;
  Items:=TObjectList.Create();

  CoInitializeEx(nil,COINIT_APARTMENTTHREADED or COINIT_DISABLE_OLE1DDE);

  try
    // Create an instance of the firewall settings manager.
    hr := CoCreateInstance(
            CLASS_NetFwMgr,
            nil,
            CLSCTX_INPROC_SERVER,
            IID_INetFwMgr,
            fwMgr
            );
    if (FAILED(hr)) then
      raise Exception.Create(Format('CoCreateInstance failed: 0x%8x',[hr]));

    // Retrieve the local firewall policy.
    fwPolicy:=fwMgr.LocalPolicy;
    if (fwPolicy=nil) then
      raise Exception.Create('get_LocalPolicy failed');

    // Retrieve the firewall profile currently in effect.
    fwProfile:=fwPolicy.CurrentProfile;
    if (fwProfile=nil) then
      raise Exception.Create('get_CurrentProfile failed');

  except
    fwProfile:=nil;
  end;
end;
{-------------------------------------------------------------------------------}
destructor TWindowsFirewall.Destroy();
var        ptr : Pointer; 
begin
  if (FAutoRollback) then
    for ptr in Items do
      RemoveItem(TItem(ptr));
  Items.Free();

  CoUninitialize();
  inherited;
end;
{-------------------------------------------------------------------------------}
function TWindowsFirewall.GetEnabled () : Boolean;
begin
  if fwProfile<>nil then
    Result:=fwProfile.FirewallEnabled
  else
    Result:=false;
end;
{-------------------------------------------------------------------------------}
procedure TWindowsFirewall.SetEnabled (Value : Boolean);
begin
  if (fwProfile<>nil) then begin
    if (fwProfile.FirewallEnabled<>Value) then
      fwProfile.FirewallEnabled:=Value;
  end;
end;
{-------------------------------------------------------------------------------}
function TWindowsFirewall.AppIsEnabled (const ProcessImageFileName : WideString) : Boolean;
var      hr : HRESULT;
         fwBstrProcessImageFileName : PWideChar;
         fwApp : INetFwAuthorizedApplication;
         fwApps : INetFwAuthorizedApplications;

begin
  hr:=S_OK;
  fwApp:=nil;
  fwApps:=nil;

  Result:=false;
  if (fwProfile=nil) then
    exit;

  // Retrieve the authorized application collection.
  fwApps:=fwProfile.AuthorizedApplications;

  if (fwApps=nil) then
      raise Exception.Create(Format('get_AuthorizedApplications failed: 0x%x',[hr]));

  // Allocate a BSTR for the process image file name.
  fwBstrProcessImageFileName:=SysAllocString(PWideChar(ProcessImageFileName));
  try
    if (fwBstrProcessImageFileName=nil) then begin
        hr := E_OUTOFMEMORY;
        raise Exception.Create(Format('SysAllocString failed: 0x%x',[hr]));
    end;

    // Attempt to retrieve the authorized application.
    try
      fwApp:=fwApps.Item(fwBstrProcessImageFileName);
      Result:=fwApp.Enabled
    except
      // The authorized application was not in the collection.
      Result:=false;
    end;
  finally
    SysFreeString(fwBstrProcessImageFileName);
  end;
end;
{-------------------------------------------------------------------------------}
procedure TWindowsFirewall.RemoveApp (const ProcessImageFileName : WideString);
var      hr : HRESULT;
         fwBstrProcessImageFileName : PWideChar;
         fwApps : INetFwAuthorizedApplications;
begin
  if (fwProfile=nil) then
    exit;

  if (AppIsEnabled(ProcessImageFileName)) then begin
    hr:=S_OK;
    fwApps:=nil;

    // Retrieve the authorized application collection.
    fwApps:=fwProfile.AuthorizedApplications;

    if (fwApps=nil) then
        raise Exception.Create(Format('get_AuthorizedApplications failed: 0x%x',[hr]));

    // Allocate a BSTR for the process image file name.
    fwBstrProcessImageFileName:=SysAllocString(PWideChar(ProcessImageFileName));
    try
      if (fwBstrProcessImageFileName=nil) then begin
          hr := E_OUTOFMEMORY;
          raise Exception.Create(Format('SysAllocString failed: 0x%x',[hr]));
      end;

      // Attempt to retrieve the authorized application.
      try
        fwApps.Remove(fwBstrProcessImageFileName);
      except
      end;
    finally
      SysFreeString(fwBstrProcessImageFileName);
    end;
  end;
end;
{-------------------------------------------------------------------------------}
procedure TWindowsFirewall.AddApp(const ProcessImageFileName, Name : WideString; AScope : TScope = wfsAll);
var      hr :HRESULT;
         fwBstrName : PWideChar;
         fwBstrProcessImageFileName : PWideChar;
         fwApp : INetFwAuthorizedApplication;
         fwApps : INetFwAuthorizedApplications;

begin
  if (fwProfile=nil) then
    exit;

  hr := S_OK;
  fwBstrName:=nil;
  fwBstrProcessImageFileName:=nil;
  fwApp:=nil;
  fwApps:=nil;
  try
    // Only add the application if it isn't already authorized.
    if (not AppIsEnabled(ProcessImageFileName)) then begin
      // Retrieve the authorized application collection.
      fwApps:=fwProfile.AuthorizedApplications;
      if (fwApps=nil) then
        raise Exception.Create(Format('get_AuthorizedApplications failed: 0x%x',[hr]));

      // Create an instance of an authorized application.
      hr := CoCreateInstance(
              CLASS_NetFwAuthorizedApplication,
              nil,
              CLSCTX_INPROC_SERVER,
              INetFwAuthorizedApplication,
              fwApp
              );
      if (FAILED(hr)) then
        raise Exception.Create(Format('CoCreateInstance failed: 0x%x',[hr]));

      // Allocate a BSTR for the process image file name.
      fwBstrProcessImageFileName := SysAllocString(PWideChar(ProcessImageFileName));
      if (fwBstrProcessImageFileName = nil) then begin
        hr := E_OUTOFMEMORY;
        raise Exception.Create(Format('SysAllocString failed: 0x%x',[hr]));
      end;

      // Set the process image file name.
      fwApp.ProcessImageFileName:=fwBstrProcessImageFileName;

      // Allocate a BSTR for the application friendly name.
      fwBstrName := SysAllocString(PWideChar(Name));
      if (SysStringLen(fwBstrName) = 0) then
      begin
        hr := E_OUTOFMEMORY;
        raise Exception.Create(Format('SysAllocString failed: 0x%x',[hr]));
      end;

      // Set the application friendly name.
      fwApp.Name:=fwBstrName;
      case AScope of
        wfsAll         : fwApp.Scope:=NET_FW_SCOPE_ALL;
        wfsLocalSubnet : fwApp.Scope:=NET_FW_SCOPE_LOCAL_SUBNET;
        wfsLocalhost   : begin
                           fwApp.Scope:=NET_FW_SCOPE_CUSTOM;
                           fwApp.RemoteAddresses:='127.0.0.1/32';
                         end;  
      end;
      
      // Add the application to the collection.
      fwApps.Add(fwApp);


      if (FindAppItem(ProcessImageFileName)=nil) then
        Items.Add(TApplication.Create(ProcessImageFileName,Name));
    end;
  finally
    SysFreeString(fwBstrName);
    SysFreeString(fwBstrProcessImageFileName);
  end;
end;
{-------------------------------------------------------------------------------}
function TWindowsFirewall.PortIsEnabled (portNumber : Integer; ipProtocol : NET_FW_IP_PROTOCOL_) : Boolean;
var      fwOpenPort : INetFwOpenPort;
         fwOpenPorts : INetFwOpenPorts;
begin
  Result:=false;
  if (fwProfile=nil) then
    exit;

  // Retrieve the globally open ports collection.
  fwOpenPorts:=fwProfile.GloballyOpenPorts;
  if (fwOpenPorts=nil) then
    raise Exception.Create('get_GloballyOpenPorts failed');

    // Attempt to retrieve the globally open port.
  try
    fwOpenPort:=fwOpenPorts.Item(portNumber, ipProtocol);
    if (fwOpenPort<>nil) then
      Result:=fwOpenPort.Enabled;
  except
    Result:=false;
  end;
end;
{-------------------------------------------------------------------------------}
function TWindowsFirewall.AddPort (portNumber : Integer; ipProtocol : NET_FW_IP_PROTOCOL_; Name : WideString; LocalScope : Boolean = false) : TPort;
var       hr : HRESULT;
          fwBstrName : PWideChar;
          fwOpenPort : INetFwOpenPort;
          fwOpenPorts : INetFwOpenPorts;
begin
  Result:=nil;
  if (fwProfile=nil) then
    exit;
    
  // Only add the port if it isn't already added.
  if (not PortIsEnabled(portNumber, ipProtocol)) then begin
    // Retrieve the collection of globally open ports.
    fwOpenPorts:=fwProfile.GloballyOpenPorts;
    if (fwOpenPorts=nil) then
      raise Exception.Create('get_GloballyOpenPorts failed');

    // Create an instance of an open port.
    hr := CoCreateInstance(
            CLASS_NetFwOpenPort,
            nil,
            CLSCTX_INPROC_SERVER,
            INetFwOpenPort,
            fwOpenPort
            );
    if (FAILED(hr)) then
        raise Exception.Create(Format('CoCreateInstance failed: 0x%x',[hr]));

    // Set the port number.
    fwOpenPort.Port:=portNumber;

    // Set the IP protocol.
    fwOpenPort.Protocol:=ipProtocol;

    // Allocate a BSTR for the friendly name of the port.
    fwBstrName := SysAllocString(PWideChar(name));
    try
      if (SysStringLen(fwBstrName) = 0) then begin
        hr := E_OUTOFMEMORY;
        raise Exception.Create(Format('SysAllocString failed: 0x%x',[hr]));
      end;

      // Set the friendly name of the port.
      fwOpenPort.Name:=fwBstrName;

      if (LocalScope) then
        fwOpenPort.Scope:=NET_FW_SCOPE_LOCAL_SUBNET;

      // Opens the port and adds it to the collection.
      fwOpenPorts.Add(fwOpenPort);

      if (FindPortItem(portNumber,ipProtocol)=nil) then
        Items.Add(TPort.Create(portNumber,ipProtocol,Name));

      Result:=FindPortItem(portNumber,ipProtocol);
    finally
      SysFreeString(fwBstrName);
    end;
  end;
end;
{-------------------------------------------------------------------------------}
procedure TWindowsFirewall.RemovePort(portNumber: Integer;
  ipProtocol: NET_FW_IP_PROTOCOL_);
var      fwOpenPorts : INetFwOpenPorts;
begin
  if (fwProfile=nil) then
    exit;

  if PortIsEnabled(portNumber,ipProtocol) then begin

    // Retrieve the globally open ports collection.
    fwOpenPorts:=fwProfile.GloballyOpenPorts;
    if (fwOpenPorts=nil) then
      raise Exception.Create('get_GloballyOpenPorts failed');

    // Attempt to remove the globally open port.
    try
      fwOpenPorts.Remove(portNumber, ipProtocol);
    except
    end;
  end;
end;
{-------------------------------------------------------------------------------}
procedure TWindowsFirewall.RemovePort (APortItem : TPort);
begin
  if (fwProfile=nil) then
    exit;

  if (APortItem<>nil) then begin
    RemoveItem(APortItem);
    Items.Remove(APortItem);
  end;
end;
{-------------------------------------------------------------------------------}
procedure TWindowsFirewall.RemoveItem(AItem: TWindowsFirewall.TItem);
begin
  if (fwProfile=nil) then
    exit;

  if (AItem is TPort) then begin
    RemovePort(TPort(AItem).PortNumber,TPort(AItem).ipProtocol);
  end else if (AItem is TApplication) then begin
    RemoveApp(TApplication(AItem).ProcessImageFileName);
  end else
    raise Exception.Create('Cannot remove unknown item');
end;
{-------------------------------------------------------------------------------}
function TWindowsFirewall.FindPortItem(portNumber: Integer; ipProtocol: NET_FW_IP_PROTOCOL_): TPort;
var      ptr : Pointer;
begin
  Result:=nil;
  if (fwProfile=nil) then
    exit;

  for ptr in ITems do begin
    if TItem(ptr) is TPort then begin
      if (TPort(ptr).PortNumber=portnumber)
         and (TPort(ptr).ipProtocol=ipProtocol) then begin
        Result:=ptr;
        exit;
      end;
    end;
  end;
end;
{-------------------------------------------------------------------------------}
function TWindowsFirewall.FindAppItem(AProcessImageFileName: WideString): TApplication;
var      ptr : Pointer;
begin
  Result:=nil;
  if (fwProfile=nil) then
    exit;

  for ptr in ITems do begin
    if TItem(ptr) is TApplication then begin
      if (TApplication(ptr).ProcessImageFileName=AProcessImageFileName) then begin
        Result:=ptr;
        exit;
      end;
    end;
  end;
end;
{-------------------------------------------------------------------------------}
procedure TWindowsFirewall.RollBack;
var       ptr : Pointer;
begin
  for ptr in Items do
    RemoveItem(TItem(ptr));
  Items.Clear();
end;
{-------------------------------------------------------------------------------}
{ TWindowsFirewall.TItem }
{-------------------------------------------------------------------------------}
constructor TWindowsFirewall.TItem.Create(AName: WideString);
begin
  inherited Create();
  Name:=AName;
  Enabled:=true;
end;
{-------------------------------------------------------------------------------}
{ TWindowsFirewall.TApplication }
{-------------------------------------------------------------------------------}
constructor TWindowsFirewall.TApplication.Create(
  AProcessImageFileName, AName: WideString);
begin
  inherited Create (AName);
  ProcessImageFileName:=AProcessImageFileName;
end;
{-------------------------------------------------------------------------------}
{ TWindowsFirewall.TPort }
{-------------------------------------------------------------------------------}
constructor TWindowsFirewall.TPort.Create(APortNumber: Integer;
  AnIpProtocol: NET_FW_IP_PROTOCOL_; AName: WideString);
begin
  inherited Create(AName);
  PortNumber:=APortNumber;
  ipProtocol:=AnIpProtocol;
end;
{-------------------------------------------------------------------------------}
end.

